using System;

using gov.va.med.vbecs.BOL;

namespace gov.va.med.vbecs.admin.GUI
{
	/// <summary>
	/// Base division-object bound control.
	/// </summary>
	public 
#if !DESIGN // Making class not abstract in DESIGN configuration allows editing the form (and derived forms ;) in VS.NET designer
	abstract 		
#endif		
	class BaseDivisionBoundControl : BaseAdminFormControl
	{	
		private DivisionBase _boundDivision;
		private BusinessObjectGuiValidator _inputValidator;
		private bool _onLoadInitDone;

		/// <summary>
		/// ControlDataChanged Event
		/// </summary>
		public event EventHandler ControlDataChanged;

		/// <summary>
		/// Constructor
		/// </summary>
		protected BaseDivisionBoundControl()
		{
			_onLoadInitDone = false;
		}

		/// <summary>
		/// Constructor
		/// </summary>
		/// <param name="brokenRulesClassName"></param>
		public BaseDivisionBoundControl( string brokenRulesClassName ) 
			: base() 
		{
			_boundDivision = null;
			_inputValidator = new BusinessObjectGuiValidator( erpMainProvider, brokenRulesClassName );
		}

		// This method is added to get around the specifics of tab control 
		// which calls control's OnLoad only when they're shown on a tab page.

		/// <summary>
		/// This method is added to get around the specifics of tab control 
		/// which calls control's OnLoad only when they're shown on a tab page.
		/// </summary>
		public void DoOnLoadInit() 
		{
			if( _onLoadInitDone ) 
				return;

			_onLoadInitDone = true;

			OnLoadInit();			
		}
	
		/// <summary>
		/// Load Event
		/// </summary>
		/// <param name="e"></param>
		protected override void OnLoad( EventArgs e )
		{
			base.OnLoad( e );
			
			DoOnLoadInit();
		}

		/// <summary>
		/// Derived classes are expected to override this method 
		/// to be used instead of OnLoad.
		/// </summary>
		protected virtual void OnLoadInit()
		{
			BindValidator();
			ResetToNoDivision();
		}

		/// <summary>
		/// BindToDivision
		/// </summary>
		/// <param name="division"></param>
		public virtual void BindToDivision( DivisionBase division )
		{
			if( division == null )
				throw( new ArgumentNullException( "division" ) );

			_boundDivision = division;
			
			DataLoadMode.Lock();
			
			DisplayData();
			OnDataChanged( true );

			DataLoadMode.Release();
		}

		/// <summary>
		/// ResetToNoDivision
		/// </summary>
		public virtual void ResetToNoDivision()
		{
			_boundDivision = null;

			ResetUiToNoSelection();
		}

		/// <summary>
		/// DisplayData
		/// </summary>
		protected 
#if !DESIGN 
		abstract void DisplayData();
#else
		virtual void DisplayData() {}
#endif			

		/// <summary>
		/// ResetUiToNoSelection
		/// </summary>
		protected 
#if !DESIGN 
		abstract void ResetUiToNoSelection();
#else
		virtual void ResetUiToNoSelection() {}
#endif			

		/// <summary>
		/// BindValidator
		/// </summary>
		protected 
#if !DESIGN 
		abstract void BindValidator();
#else
		virtual void BindValidator() {}
#endif			

		/// <summary>
		/// TransferUserInputToBoundDivisionObject
		/// </summary>
		protected 
#if !DESIGN 
		abstract void TransferUserInputToBoundDivisionObject();
#else
		virtual void TransferUserInputToBoundDivisionObject() {}
#endif	

		/// <summary>
		/// Get BusinessObjectUnderValidation
		/// </summary>
		protected
#if !DESIGN
		abstract BaseBusinessObject BusinessObjectUnderValidation
		{
			get;
		}
#else
		virtual BaseBusinessObject BusinessObjectUnderValidation
		{
			get
			{
				return null; // stub value
			}
		}
#endif
		/// <summary>
		/// ConfirmExit
		/// </summary>
		/// <returns></returns>
		public virtual bool ConfirmExit()
		{
			return true;
		}

		/// <summary>
		/// NumericUpDown_KeyUp
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		protected virtual void NumericUpDown_KeyUp(object sender, System.Windows.Forms.KeyEventArgs e)
		{
			DataEntryFields_DataChanged( sender, e );
		}

		/// <summary>
		/// DataEntryFields_DataChanged
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		protected void DataEntryFields_DataChanged( object sender, System.EventArgs e )
		{
			if( DataLoadMode.IsLocked ) // To prevent processing of events when data is being transferred to widgets.
				return;

			OnDataChanged( false );
		}

		/// <summary>
		/// DataChanged Event
		/// </summary>
		/// <param name="dontTransferInput"></param>
		protected virtual void OnDataChanged( bool dontTransferInput ) 
		{
			if( !dontTransferInput )
				TransferUserInputToBoundDivisionObject();

			InputValidator.Validate( BusinessObjectUnderValidation );

			if( ControlDataChanged != null )
				ControlDataChanged( this, EventArgs.Empty );
		}

		/// <summary>
		/// Get BoundDivision
		/// </summary>
		protected virtual DivisionBase BoundDivision
		{
			get
			{
				return _boundDivision;
			}
		}

		/// <summary>
		/// Get InputValidator
		/// </summary>
		protected virtual BusinessObjectGuiValidator InputValidator
		{
			get
			{
				return _inputValidator;
			}
		}
	}
}
